//
//  main.swift
//  MacOSPlayGroundDemo
//
//  Created by CAEN RFID on 07/07/23.
//

import Foundation
import RFIDLibrary

// Output text
var outputText:String = ""
var pressed = false

//Queue where to launch all rfid api calls.
let queue:DispatchQueue = DispatchQueue.init(label: "", qos: .default)
//Instance of CAENRFIDReader
let reader:CAENRFIDReader? = CAENRFIDReader()
//Inventory semaphore used to stop api calls on continuous inventory for 2 seconds.
let inventoryWaiter:DispatchSemaphore = DispatchSemaphore.init(value: 0)

// Utils extension for Data type to print hexadecimal string
extension Data {
    struct HexEncodingOptions: OptionSet {
        let rawValue: Int
        static let upperCase = HexEncodingOptions(rawValue: 1 << 0)
    }
    
    func hexEncodedString(options: HexEncodingOptions = []) -> String {
        let format = options.contains(.upperCase) ? "%02hhX" : "%02hhx"
        return self.map { String(format: format, $0) }.joined()
    }
}

//Callback set for continuous inventory operation.
func onTags(args:CAENRFIDEventArgs, resultCode:UInt16?) -> Void {
    guard (resultCode == nil) else {
        outputText.append("\nError in continuous inventory -> \(resultCode ?? UInt16.max)\n")
        inventoryWaiter.signal()
        return
    }
    args.data?.forEach{ tag in
        let rssi = Double(tag.getRSSI()/10)
        let tagId = Data(tag.getId()).hexEncodedString()
        outputText.append("Continuous Tag -> \(tagId) e RSSI: \(rssi) dBm.\n")
    }
}

//MAIN
do{
    //Clean output text
    outputText.removeAll(keepingCapacity: true)
    //1) Connect to reader. Depending on connection type, you should remember to add
    //   right entitlements in info plist or capabilities in sandbox. See info.plist
    //   for the needed entitlements.
    
    //_ = try reader?.connect(.caenrfid_rs232, "/dev/tty.usbmodem11403:115200:N:8:1")
    _ = try reader?.connect(.caenrfid_tcp, "10.106.254.115")
    //2) Check if connection was closed abruptly during an inventory continuous for 3 seconds.
    let wasIncontinuous:Bool? = try reader?.forceAbort(timeout: 3000000000)
    if wasIncontinuous ?? false {
        outputText.append("Reader was in conitnuous inventory\n")
    }
    let info:CAENRFIDReaderInfo? = try reader?.getReaderInfo()
    let fwRelease:String? = try reader?.getFWRelease()
    let regulation:CAENRFIDRFRegulations? = try reader?.getRFRegulation()
    outputText.append("Reader info are:\n")
    outputText.append("  MODEL      : " + (info?.model ?? "unknown"))
    outputText.append("\n  SERIAL     : " + (info?.serialNumber ?? "unknown"))
    outputText.append("\n  FW Rel.    : " + (fwRelease ?? "unknown"))
    outputText.append("\n  Regulation : \(regulation ?? .UNKNOWN )")
    let power = try reader?.getPower()
    outputText.append("\n  Conducted Power (in mW): \(power ?? -1)")
    let airLinkProfile = try reader?.getBitRate()
    outputText.append("\n  Airlink profile : \(airLinkProfile ?? .UNKNOWN)")
    
    // 4) Set 50 mW of conducted power, and get first logical source (default source) instance.
    _ = try reader?.SetPower(50)
    let source:CAENRFIDLogicalSource? = try? reader?.getSource("Source_0")
    outputText.append("\nSource " + (source?.getName() ?? "no name") )
    // 5) Launch Continuous inventory with RSSI flag.
    let flags:UInt16 = CAENRFIDLogicalSource.InventoryFlag.CONTINUOUS.rawValue + CAENRFIDLogicalSource.InventoryFlag.FRAMED.rawValue + CAENRFIDLogicalSource.InventoryFlag.RSSI.rawValue
    //6) Configure Continuous inventory callback
    _ = reader?.setCAENRFIDEvent(onTags)
    //7) Set Inventory read cycles of the logical source to 0 (infinite cycles)
    _ = try source?.setReadCycle(0)
    //8) Start Continuous inventory (10 seconds)
    let isStarted:Bool = try source?.eventInventoryTag(flag: flags) ?? false
    //9) If started, the continuous inventory will let the reader start a stream of tags
    //   each time the antenna will discovery one. The print job is done by the callback.
    if isStarted {
        // 10) Wait 2 seconds to let the reader identify some tags in the meanwhile.
        _ = inventoryWaiter.wait(timeout: .now() + 2)
        // 11) After that, stops the inventory continuous.
        try reader?.inventoryAbort()
    } else {
        outputText.append("\nNo continuous inventory has been started")
    }
}catch CAENRFIDError.communicationError(let desc, _){
    outputText.append("Comm Errore" + desc + "\n")
}catch CAENRFIDError.libraryError(let desc) {
    outputText.append("Lib Errore" + desc + " \n")
}catch CAENRFIDError.operationError(let desc, _) {
    outputText.append("LibOperation Errore" + desc + "\n")
}catch{
    outputText.append("GENERAL ERROR \(error) \n")
}
//12) Disconnect the reader
do{
    try reader?.disconnect()
    outputText.append("Disconnection done")
    
}catch {
    print("Disconnection error")
}
// ---------------------------/
// PRINT ALL OUTPUT TO CONSOLE
print(outputText)
//----------------------------/
